%% Description
% Plots figure 6: dimensionless experimental data for the force and drop
% radius compared to the theoretical equilibrium solutions.

%% Experimental parameter definitions
%
% L             Radius of membrane (m).
%
% g             Surface tension parameter: \gamma \cos \theta (N/m).
%
% V             Volume of droplet in each experiment (uL).
%
% T             Membrane tension in each experiment (N/m).
%
% t             Time in each experiment (s).
%
% H             Clamp height as function of time for each experiment (mm).
%
% F             Force as function of time for each experiment (mN).
%
% R             Droplet radius (fit from video) as function of time for 
%               each experiment (mm).

%% Theoretical parameters
%
% h             Values of Hinf in theoretical data.
%
% f             Theoretical force for matching experimental data.
%
% r             Theoretical droplet radius for matching experimental data.
%
% col           Colour scheme for plotting.

%% Load solution
load('figure6.mat')

%% Plot (dimensionless) experimental data
% Colour Scheme
col = parula(3);

Hconv = V.*1e-6./L^2;     % height conversion factor (V/L^2)
fconv = g*L^4./V./1e-12;  % force conversion factor (g L^4/V)
Rconv = L/1e-3;

t1 = t1(1:20:end);
t2 = t2(1:20:end);
t3 = t3(1:20:end);

% Plot adhesion force (fig 6a)
figure(1)
loglog(H1(t1)./Hconv(1),F1(1:20:end)./fconv(1),...
            'color',col(1,:),'marker','s','linestyle','none'), hold on
        
loglog(H2(t2)./Hconv(2),F2(1:20:end)./fconv(2),...
            'color',col(2,:),'marker','x','linestyle','none')
        
loglog(H3(t3)./Hconv(3),F3(1:20:end)./fconv(3),...
            'color',col(3,:),'marker','^','linestyle','none')
        
xlabel('$H_\infty$','interpreter','latex')
ylabel('$F$','interpreter','latex')

set(findobj('type','axes'),'fontsize',20,'linewidth',2)
set(gca,'TickLength',2*get(gca,'TickLength'))
axis([5 1e2 1e-4 1])
 
% Plot droplet radius (fig 6b)
figure(2)
loglog(H1(t1)./Hconv(1),R1(1:20:end)./Rconv,...
            'color',col(1,:),'marker','s','linestyle','none'), hold on
        
loglog(H2(t2)./Hconv(2),R2(1:20:end)./Rconv,...
            'color',col(2,:),'marker','x','linestyle','none')
        
loglog(H3(t3)./Hconv(3),R3(1:20:end)./Rconv,...
            'color',col(3,:),'marker','^','linestyle','none')

xlabel('$H_\infty$','interpreter','latex')
ylabel('$R_M$','interpreter','latex')
axis([5 1e2 0.05 1])
set(findobj('type','axes'),'fontsize',20,'linewidth',2)
set(gca,'TickLength',2*get(gca,'TickLength'))

%% Plot theoretical data
for k=1:3
    % Need to join between types of solutions in plot
    % Find where F non-zero first/last for each type of solution
        n1 = find(f(:,1,k),1,'first');
        n2 = find(f(:,2,k),1,'first');
        n3 = find(f(:,2,k),1,'last');
        n4 = find(f(:,3,k),1,'last');
        
        figure(1)
        loglog([mean([h(n1) h(n2)]); h(n1:end)],...
            [mean([f(n1,1,k) f(n2,2,k)]); f(n1:end,1,k)],...
            'color',col(k,:),'linewidth',2)
        loglog([mean([h(n1) h(n2)]); h(n2:n3); mean([h(n3) h(n4)])],...
            [mean([f(n1,1,k) f(n2,2,k)]); f(n2:n3,2,k); mean([f(n3,2,k) f(n4,3,k)])],...
            'color',col(k,:),'linewidth',2,'linestyle','--')
        loglog([h(1:n4); mean([h(n3) h(n4)])],...
            [f(1:n4,3,k); mean([f(n3,2,k) f(n4,3,k)])],...
            'color',col(k,:),'linewidth',4)
        
        figure(2)
        loglog([mean([h(n1) h(n2)]); h(n1:end)],...
            [mean([r(n1,1,k) r(n2,2,k)]); r(n1:end,1,k)],...
            'color',col(k,:),'linewidth',2)
        loglog([mean([h(n1) h(n2)]); h(n2:n3); mean([h(n3) h(n4)])],...
            [mean([r(n1,1,k) r(n2,2,k)]); r(n2:n3,2,k); mean([r(n3,2,k) r(n4,3,k)])],...
            'color',col(k,:),'linewidth',2,'linestyle','--')
        loglog([h(1:n4); mean([h(n3) h(n4)])],...
            [r(1:n4,3,k); mean([r(n3,2,k) r(n4,3,k)])],...
            'color',col(k,:),'linewidth',4)
        
        
%     figure(1)
%     loglog(h,f(:,1,k),'color',col(k,:),'linewidth',2), hold on
%     loglog(h,f(:,2,k),'color',col(k,:),'linewidth',2,'linestyle','--')
%     loglog(h,f(:,3,k),'color',col(k,:),'linewidth',4)
    
%     figure(2)
%     loglog(h,r(:,1,k),'color',col(k,:),'linewidth',2), hold on
%     loglog(h,r(:,2,k),'color',col(k,:),'linewidth',2,'linestyle','--')
%     loglog(h,r(:,3,k),'color',col(k,:),'linewidth',4)
end

